oxfordize <- function(string_vec) {
  if (length(string_vec) == 1)
    string_vec
  else if (length(string_vec) == 2)
    paste(string_vec, collapse = " and ")
  else {
    if (!requireNamespace("pander", quietly = TRUE)) install.packages("pander", quiet = TRUE)
    pander::p(string_vec, wrap = "", copula = ", and ")
  }
}

within_n_mads <- function(vals, n = 3) {
  valmad <- mad(vals, na.rm = TRUE)
  valmed <- median(vals,na.rm = TRUE)
  valadj <- n * valmad
  low <- valmed - valadj
  high <- valmed + valadj
  between(vals, low, high)
}

# median_scale <- function(x) {
#   x <- as.matrix(x)
#   fun <- function(vec) {(vec - median(vec, na.rm = TRUE))/mad(vec, na.rm = TRUE)}
#   out <- apply(x, 2, fun)
#   out
# }

# # Extract from a data frame (or named vector) the variables listed in a formula
# extract_vars <- function(formula, df) {
#   df[names(df) %in% rownames(attr(terms.formula(formula), "factors"))]
# }

pairs.panels <- function (df, method = "spearman", pch = 20, 
                          hist.col = "#21908CFF",  
                          cex.cor = 10, ...) {
  
  "panel.hist.density" <- function(x, ...) {
    usr <- par("usr")
    on.exit(par(usr))
    par(usr = c(usr[1:2], 0, 1.5))
    h <- hist(x, plot = FALSE)
    breaks <- h$breaks
    nB <- length(breaks)
    y <- h$counts
    y <- y/max(y)
    rect(breaks[-nB], 0, breaks[-1], y, col = hist.col)
    tryd <- try(d <- density(x, na.rm = TRUE, bw = "nrd", 
                             adjust = 1.2), silent = TRUE)
    if (class(tryd) != "try-error") {
      d$y <- d$y/max(d$y)
      lines(d)
    }
  }
  
  "panel.cor" <- function(x, y, digits = 2, prefix = "", ...) {
    usr <- par("usr")
    on.exit(par(usr))
    par(usr = c(0, 1, 0, 1))
    r <- cor(x, y, use = "pairwise", method = method)
    # Add background fill
    colors <- colorRampPalette(c("#053061", "#2166AC", "#4393C3", "#92C5DE", 
                                 "#D1E5F0", "#FFFFFF", "#FDDBC7", "#F4A582", 
                                 "#D6604D", "#B2182B", "#67001F"))(200)
    col.fill <- colors[findInterval(r, seq(-1, 1, length.out = 200))]
    # Scale box size
    r.range <- seq(0, 1, length.out = 200)
    fancy.size <- 4/3 - 4/3*exp(-log(2)/0.5 * r.range)
    adj <- fancy.size[findInterval(abs(r), r.range)] * 0.5
    polygon(x = c(0.5-adj, 0.5+adj, 0.5+adj, 0.5-adj),
            y = c(0.5+adj, 0.5+adj, 0.5-adj, 0.5-adj),
            border = col.fill, col = col.fill)
    txt <- format(c(round(r, 2), 0.123456789), digits = 2)[1]
    txt <- paste(prefix, txt, sep = "")
    cex <- cex.cor * 0.8/strwidth(txt)
    cex1 <- cex * adj * 2
    if (cex1 < 0.25) cex1 <- 0.25
    text(0.5, 0.5, txt, cex = cex1)
  }
  
  "panel.smoother" <- function(x, y, pch = par("pch"), 
                               col.smooth = "red", span = 2/3, iter = 3, ...) {
    xm <- mean(x, na.rm = TRUE)
    ym <- mean(y, na.rm = TRUE)
    xs <- sd(x, na.rm = TRUE)
    ys <- sd(y, na.rm = TRUE)
    r = cor(x, y, use = "pairwise", method = method)
    
    points(x, y, pch = pch, ...)
    ok <- is.finite(x) & is.finite(y)
    if (any(ok)) 
      lines(stats::lowess(x[ok], y[ok], f = span, iter = iter), 
            col = col.smooth, ...)
  }
  
  old.par <- par(no.readonly = TRUE)
  on.exit(par(old.par))
  if (missing(cex.cor)) 
    cex.cor <- 1
  for (i in 1:ncol(df)) {
    if (is.character(df[[i]])) {
      df[[i]] <- as.numeric(as.factor(df[[i]]))
      colnames(df)[i] <- paste(colnames(df)[i], "*", sep = "")
    }
  }
  pairs(df, diag.panel = panel.hist.density, lower.panel = panel.cor, 
        upper.panel = panel.smoother, 
        pch = pch, ...)
}

summarize_vars <- function(dat, ...) {
  check_vars <- select(dat, ...)
  if (!all(sapply(check_vars, class) %in% c("integer", "numeric")))
    stop("Summary stats only available for numerical variables.")
  means_sds <- dat %>%
    pivot_longer(..., names_to = "variable") %>%
    group_by(variable) %>%
    summarise(mean = mean(value, na.rm = TRUE),
              sd = sd(value, na.rm = TRUE))
}

add_sun <- function (obs_data) {
  date_range <- range(as.Date(obs_data$DateTime))
  start <- date_range[1]; end <- date_range[2]
  direction <- "sunrise"
  out_tz <- "America/New_York"
  lon <- -80.2367
  lat <- 34.58234
  
  if (!requireNamespace("maptools", quietly = TRUE)) 
    install.packages("maptools", quiet = TRUE)
  
  ll <- cbind(lon, lat)
  start <- as.POSIXct(as.character(start), tz = out_tz)
  end <- as.POSIXct(as.character(end), tz = out_tz)
  sequence <- seq(from = start, to = end, by = "days")
  sun <- lapply(direction, function(d) {
    if (d %in% c("sunrise", "sunset")) 
      tmp <- maptools::sunriset(ll, sequence, direction = d, 
                                POSIXct = TRUE)$time
    else tmp <- maptools::crepuscule(ll, sequence, solarDep = 6, 
                                     direction = d, POSIXct = TRUE)$time
    if (out_tz != "GMT") {
      attributes(tmp)$tzone <- out_tz
    }
    tmp <- data.frame(date = as.Date(sequence), 
                      tmp, stringsAsFactors = FALSE)
    names(tmp)[2] <- d
    tmp
  })
  sun <- Reduce(function(...) merge(..., by = "date", 
                                    all.x = TRUE), sun)
  obs_data <- mutate(obs_data, date = as.Date(DateTime))
  obs_data <- left_join(obs_data, sun, by = "date") %>%
    mutate(since_rise_h = round(as.numeric(difftime(DateTime, sunrise, units = "hours")), 3)) %>%
    select(-date, -sunrise)
  return(obs_data)
}

# Create data frame of n columns
combinations <- function(n, values = 0) {
  expand.grid(rep(list(values), n))
}

date_seq <- function(st = "2020-01-01", end = "2020-06-30", by = 1, format = "%d %b") {
  out <- seq.Date(as_date(st), as_date(end), by = by)
  return(format(out, format = format))
}

extract_dates <- function(doys) {
  date_seq <- date_seq()
  out <- date_seq[doys]
  return(out)
}

dfc <- function(cov_vals, var, all_vars, var_summary, include_year = FALSE, 
                year_only = FALSE, display_only = FALSE) {
  if (year_only)
    known_dfc <- data.frame(variable = "year",
                            dfc = floor(year(now()) / 2) * 2)
  else {
    known_dfc <- data.frame(variable = c("pine_ba", "canopy", "burns_10_freq"),
                            dfc = c(50, 0.45, 3.4))
    if (include_year)
      known_dfc <- bind_rows(known_dfc,
                             data.frame(variable = "year",
                                        dfc = floor(year(now()) / 2) * 2))
  }

  if (display_only) {
    message("Current values specified for desired future conditions:")
    return(known_dfc)
  } 
  known_dfc <- left_join(known_dfc, var_summary, by = "variable") %>%
    mutate(dfc_scaled = (dfc - mean) / sd) %>%
    select(variable, dfc_scaled) %>%
    tidyr::pivot_wider(names_from = variable, values_from = dfc_scaled)

  rem_vars <- setdiff(all_vars, c(var, names(known_dfc)))
  new_dat <- data.frame(var_scaled = cov_vals, known_dfc, combinations(length(rem_vars)))
  names(new_dat) <- c(var, names(known_dfc), rem_vars)
  return(new_dat)
}

label_lookup <- function(var_names) {
  case_when(
    var_names == "doy" ~ "Date", 
    var_names == "since_rise_h" ~ "Time since sunrise (h)", 
    var_names == "year" ~ "Survey year",
    var_names == "pine_ba" ~ "Pine basal area (sq. ft per acre)", 
    var_names == "hard_ba" ~ "Hardwood basal area (sq. ft per acre)", 
    var_names == "pine_dbh" ~ "Mean DBH of five nearest pines", 
    var_names == "snags" ~ "Number of snags within 50 m", 
    var_names == "canopy" ~ "Proportion of sampling locations with canopy present", 
    var_names == "grass" ~ "Proportion of sampling locations with grass present",
    var_names == "forb" ~ "Proportion of sampling locations with forbs present", 
    var_names == "woody" ~ "Proportion of sampling locations with woody groundcover present", 
    var_names == "bare" ~ "Proportion of sampling locations with bare ground present", 
    var_names == "litter" ~ "Proportion of sampling locations with duff/litter present", 
    var_names == "shrubs" ~ "Proportion of sampling locations with shrubs present", 
    var_names == "burns_10_freq" ~ "# burns in prior 10 years\n(fractional burns result from survey points overlapping multiple burn units)",
    TRUE ~ var_names
  )
}
