process_pre2020_bird_data <- function(pre2020_xls, flag_odd = TRUE) {
  
  years <- excel_sheets(pre2020_xls) %>%
    str_extract("^[:digit:]{4}$")
  message("Reading in bird observation data for ", oxfordize(years))
  
  # Get the data
  dat <- lapply(years, function(yr) {
    tmp <- read_excel(pre2020_xls, sheet = yr,
                      col_names = c("Date", "Point", "Time", "Observer", "Species",
                                    "Wind", "Sky", "First45", "First15",
                                    "Second45", "Second15", "Third45",
                                    "Third15", "Outside", "Total", "Notes"),
                      col_types = c(rep("text", 5), rep("numeric", 10), "text"),
                      skip = 1) %>%
      mutate(Time = as.character(chron::times(as.numeric(Time))),
             Date = as.Date(as.integer(Date), origin = "1899-12-30"),
             DateTime = as_datetime(paste(Date, Time), tz = "America/New_York"),
             Total = as.integer(First45 + First15 + Second45 + Second15 + Third45 + Third15),
             Sky = as.integer(Sky),
             Wind = as.integer(Wind),
             Observer = toupper(Observer),
             Point = toupper(Point),
             # Fix inconsistent species code for Bachman's Sparrow
             Species = ifelse(Species == "BACS", "BASP", Species))
    
    # Check for atypical detection histories
    detects <- select(tmp, Species, First45:Third15) %>%
      assertr::maha_dist(stringsAsFactors = TRUE)
    bad <- which(detects > 100)
    if (length(bad) > 0 && flag_odd) {
      message(yr, ": ", length(bad), " detection histories differed considerably",
              " from other detection histories. \nA viewer has been opened for closer inspection.")
      close_look <- tmp[bad, names(tmp) != "DateTime"]
      View(close_look, yr)
    }

    # Finish it up
    tmp <- select(tmp, DateTime, Point, Observer, Species, Wind, Sky, Total) 
    tmp
  })
  
  # Compile data and remove the limited Loggerhead Shrike data
  dat <- bind_rows(dat) %>%
    filter(Species != "LOSH")
  return(dat)
  
}