process_post2020_bird_data <- function(post2020_csv) {

  # Get the data
  dat <- read_csv(post2020_csv,
                  col_types = "------ccccc-cciiiiiiic--") %>%
    mutate(Total = as.integer(rowSums(select(., matches("15|45")))),
           Species = toupper(Species),
           Point = toupper(Point),
           Observer = toupper(Observer),
           # Accommodate an odd option for wind speed
           Wind = ifelse(grepl("choice_4", Wind), 4, Wind),
           Wind = as.integer(substr(Wind, 1, 1)),
           Sky = as.integer(substr(Sky, 1, 1)),
           DateTime = as.POSIXct(paste(gsub(" .*$", "", Date), Time), format = "%m/%d/%Y %H:%M", 
                                 tz = "America/New_York")) %>%
    relocate(Total, .before = Notes) %>%
    # Get rid of some presumed test data in the first year of digital data collection
    filter(DateTime > as_datetime("2020-03-15"))

  years <- sort(unique(year(dat$DateTime)))
  message("Reading in bird observation data for ", oxfordize(years))
  
  # Check for atypical detection histories
  detects <- select(dat, Species, matches("15|45")) %>%
    assertr::maha_dist(stringsAsFactors = TRUE)
  bad <- which(detects > 100)
  if (length(bad) > 0) {
    close_look <- dat[bad, ]
    bad_yrs <- sort(unique(year(dat$DateTime)))
    for (yr in bad_yrs) {
      close_look_i <- filter(close_look, year(DateTime) == yr) %>%
        select(-DateTime)
      message(yr, ": ", nrow(close_look_i), " detection histories differed considerably",
              " from other detection histories. \nA viewer has been opened for closer inspection.")
      View(close_look_i, as.character(yr))
    }
  }
  
  # Finish it up
  dat <- select(dat, DateTime, Point, Observer, Species, Wind, Sky, Total)
  return(dat)
  
}