process_historic_fire_data <- function(pre2022_fire_xls) {
  
  message("Reading in fire data from 2000 - 2021")
  
  # Get the data
  dat <- read_excel(pre2022_fire_xls, 
                    col_types = c("text", rep("skip", 8), "text", "numeric", 
                                  rep("skip", 6))) %>%
    mutate(start_date = as.Date(as.integer(start_date), origin = "1899-12-30"),
           bu_name = gsub("^SCCRR.*RX-|^[0-9]{4}-", "", incident_name),
           survyr_burn = ifelse(month(start_date) <= 7, year(start_date), year(start_date) + 1),
           fy_burn = ifelse(month(start_date) < 10, year(start_date), year(start_date) + 1))

  return(dat)
  
}

flag_unknown_bu <- function(fire_dat, bu_sf) {
  known_bu <- bu_sf %>%
    pull(bu_name) %>% unique() %>% sort()
  unknown_bu <- filter(fire_dat, !(bu_name %in% known_bu))
  if (nrow(unknown_bu) > 0) {
    message(nrow(unknown_bu), " burn histories have unrecognized burn unit names.",
            " A viewer has been opened for closer inspection.")
    View(unknown_bu)
  } else message("No issues found with burn history unit names!")
  unknown_bu
}

process_fy_burns <- function(burn_history, burn_units, survey_yrs) {
  # Create FY burn history shapefile for refuge
  all_burn_units <- pull(burn_units, bu_name) %>% unique() %>% sort()
  burn_yrs <- pull(burn_history, fy_burn) %>% unique() %>% sort
  burn_hist <- as.data.frame(burn_history) %>% 
    select(bu_name, fy_burn) %>%
    mutate(burned = TRUE) %>%
    complete(bu_name = all_burn_units, fy_burn = burn_yrs, fill = list(burned = FALSE)) %>%
    pivot_wider(names_from = fy_burn, values_from = burned)
  fy_burn_shp <- left_join(burn_units, burn_hist, by = "bu_name") %>%
    arrange(bu_name)
  out_shp <- glue::glue("Geodata/FY_burn_history_through_{max(burn_history$fy_burn)}.shp")
  write_sf(fy_burn_shp, out_shp, overwrite = TRUE)
  message("Updated fire history written to ", out_shp)
}

calc_survyr_burns <- function(burn_units, burn_history, survey_pts, 
                              survey_yrs, year_win = 10, buffer = 100) {
  max_year_win = min(survey_yrs) - min(burn_history$survyr_burn) + 1
  if (year_win > max_year_win) stop("Maximum year window for this data set is ", max_year_win)
  pt_buff <- st_buffer(survey_pts, buffer)
  pt_names <- sort(pull(pt_buff, NAME))
  burn_areas <- st_union(burn_units)
  pt_burn_area <- st_intersection(pt_buff, burn_areas) %>%
    select(NAME) %>%
    mutate(area = as.numeric(st_area(.))) %>%
    as.data.frame() %>% select(-geometry)
  burn_props <- st_intersection(pt_buff, burn_history) %>%
    select(NAME, survyr_burn) %>%
    mutate(burn_area = as.numeric(st_area(.))) %>%
    as.data.frame() %>% select(-geometry) %>%
    group_by(NAME, survyr_burn) %>%
    summarise(burn_area = sum(burn_area), .groups = "drop") %>%
    left_join(pt_burn_area, by = "NAME") %>%
    mutate(burn_prop = round(burn_area / area, 1)) %>%
    select(NAME, survyr_burn, burn_prop) %>%
    arrange(NAME, survyr_burn)
  burn_var <- glue::glue("burns_{year_win}_freq")
  missing_fill <- list(); missing_fill[[burn_var]] <- 0
  burn_freqs <- lapply(survey_yrs, function(yr) {
    burn_props %>%
      group_by(NAME) %>%
      filter(between(survyr_burn, yr - (year_win - 1), yr)) %>%
      summarize("{burn_var}" := sum(burn_prop)) %>%
      complete(NAME = pt_names, fill = missing_fill) %>%
      mutate(yr_assessed = yr, .after = NAME)
  })
  survyr_burn_freqs <- bind_rows(burn_freqs) %>% 
    rename(Point = NAME)
  survyr_burn_freqs
}