plot_occupancy <- function(occu_fit, type = c("state", "det"), 
                           plot_vars, var_labels = plot_vars, var_summary, 
                           species = NULL, use_dfc = FALSE, use_most_recent = FALSE){
  
  #plot_vars: covariates to plot, in desired plotting order
  #var_labels: covariate labels for the plots, in same order as plot_vars
  #var_summary: data.frame of model covariate means and sd, for putting things on original scale

  type = match.arg(type)

  # Check for plot_vars
  if (missing(plot_vars))
    stop("You must enter at least one covariate to plot.")
  if (!all(plot_vars %in% var_summary$variable))
    stop("At least one variable wasn't found in the variable summary data.")
  
  # Check names
  if (!identical(length(plot_vars), length(var_labels))) 
    stop("Mismatch between the number of variables to plot and the names for them.")
  
  plot_dat <- c(as.list(occu_fit@data@siteCovs), as.list(occu_fit@data@obsCovs))
  
  spp_menu <- c("BHNU", "BASP", "NOBO")
  if (is.null(species)) 
    species <- utils::select.list(spp_menu, 
                                  title = "What species are you plotting?", graphics = FALSE)
  if (!species %in% spp_menu) stop("Unrecognized species. Choose one of: ",
                                   paste(spp_menu, collapse = ", "))
  sp_nm <- switch(species,
                  BASP = "Bachman's Sparrow",
                  BHNU = "Brown-headed Nuthatch",
                  NOBO = "Northern Bobwhite")

  # Pull fixed covariates used in model, minus intercept
  model_vars <- lapply(unmarked:::split_formula(lme4::nobars(occu_fit@formula)), all.vars)
  state_vars <- model_vars[[2]]
  det_vars <- model_vars[[1]]
  if (type == "state") {
    if (!all(plot_vars %in% state_vars))
      stop("At least one `plot_vars` is not present in the `state` model. Check and run again.")
  }
  if (type == "det") {
    if (!all(plot_vars %in% det_vars))
      stop("At least one `plot_vars` is not present in the `detection` model. Check and run again.")
  }
  
  # List to hold ggplot objects
  figures <- vector(mode = "list", length = length(plot_vars))
  names(figures) <- plot_vars
  
  # String of all variables used in the model, not just those plotted
  all_vars <- unlist(model_vars)
  
  for (var in plot_vars) {
    
    # Specify new data for prediction and intervals
    # other covariates are held at their mean (i.e., scaled value of 0)
    cov_vals <- plot_dat[[var]]
    if (is.numeric(cov_vals)) {
      rng <- range(cov_vals, na.rm=TRUE)
      cov_vals <- seq(rng[1], rng[2], length.out = 50)
    } else if (is.factor(cov_vals)) {
      cov_vals <- factor(levels(cov_vals), levels=levels(cov_vals))
    } else {
      stop("Unknown covariate type")
    }

    if (use_dfc) {
      if (use_most_recent) 
        new_dat <- dfc(cov_vals, var, all_vars, var_summary, include_year = TRUE)
      else
        new_dat <- dfc(cov_vals, var, all_vars, var_summary, include_year = FALSE)
    } else {
      if (use_most_recent)
        new_dat <- dfc(cov_vals, var, all_vars, var_summary, year_only = TRUE)
      else {
        rem_vars <- setdiff(all_vars, var)
        new_dat <- data.frame(var_scaled = cov_vals, combinations(length(rem_vars)))
        names(new_dat) <- c(var, rem_vars)
      }
    }

    # Predicting over the range of current variable, all others at means
    new_dat <- predict(occu_fit, newdata = new_dat, type = type, re.form = NA, appendData = TRUE)
    plot_var_summary <- filter(var_summary, variable == var)
    
    # Return covariates to original scale for interpretability
    for (i in seq_along(new_dat)) {
      cur_var <- names(new_dat)[i]
      if (cur_var %in% var_summary$variable) {
        cur_var_summary <- filter(var_summary, variable == cur_var)
        new_dat[, cur_var] <- 
          (new_dat[, cur_var] * cur_var_summary$sd + cur_var_summary$mean) 
      }
    }
    
    rug_dat <- data.frame(var = plot_dat[[var]] * plot_var_summary$sd + plot_var_summary$mean,
                          Predicted = sample(new_dat$Predicted, length(plot_dat[[var]]), replace = TRUE)) # make ggplot happy
    
    new_dat <- rename(new_dat, var = glue::glue("{var}"))
    
    # Create plot
    
    # Get some parameters for labels
    xmin <- min(new_dat$var); xmax <- max(new_dat$var); x_range <- xmax - xmin
    
    p <- ggplot(new_dat, aes(x = var, y = Predicted)) + 
      geom_ribbon(aes(ymin = lower, ymax = upper), alpha = 0.3) + 
      geom_line(size = 1.25)
    
    # Tidy the plot
    p <- p + 
      labs(x = var_labels[which(plot_vars == var)],
           y = if (type == "state") "Predicted occupancy" else "Detection probability",
           title = sp_nm) +
      geom_rug(data = rug_dat, size=0.1, sides = "b", 
               position = "jitter", alpha = 0.5) + 
      theme_bw() +
      theme(legend.position = "none", plot.margin = unit(rep(0.2, 4), "cm"))
    
    # Special tweak for Day of Year variable
    if (var == "doy") {
      p <- p + scale_x_continuous(labels = extract_dates)
    }
    
    figures[[var]] <- p
  }
  
  # Create informative file name components
  year_stats <- var_summary[var_summary$variable == "year", ]
  yr_rng <- paste(range(plot_dat[["year"]] * year_stats$sd + year_stats$mean),
                  collapse = "-")
  fn_base <- paste(c(species, yr_rng, type), collapse = "_")

  # Save list of figures in case modification is needed
  saveRDS(figures, file.path("Output", paste0(fn_base, "_figures.rds")))
  
  # Save images of figures
  for (f in seq_along(figures)) {
    f_var <- names(figures)[f]
    f_fn <- file.path("Figures", paste0(fn_base, "_", gsub("_", "", f_var), ".png"))
    png(file = f_fn, width = 6.5, height = 6.5, units = "in", res = 300)
    print(figures[[f]])
    dev.off()
    message("Created plot of ", species, ifelse(type == "state", " occupancy", " detection"),
            " as a function of '", f_var, "' in ", f_fn)
  }
  return(figures)
}