# Assumes you've just worked through 00_load_and_prep_data.R
options(scipen = 999)

# Load plotting and table functions
source("Code/Functions/plot_occupancy.R")
source("Code/Functions/table_occupancy.R")

# Step though modelling plan by species
spp <- pull(occ_plan, Species) %>% unique()

spp_mods <- lapply(spp, function(sp) {
  message("Fitting occupancy model for ", sp)
  sp_years <- filter(occ_plan, Species == sp) %>%
    pull(year)
  
  ## Filter bird data
  sp_dat <- filter(bird_dat, Species == sp) %>%
    group_by(Point, year) %>%
    mutate(visit = row_number()) %>% ungroup()
  max_visits <- max(sp_dat$visit)
  
  ## Cast to detection matrix, retaining survey point ID for reference (temporarily)
  y_w_site <- select(sp_dat, Point, year, visit, det) %>%
    pivot_wider(names_from = visit, values_from = det, names_prefix = "visit") %>%
    arrange(year, Point)
  y <- as.matrix(select(y_w_site, -Point, -year))
  
  # Naive occupancy estimate
  # Numerator sums the number of sites with at least 1 detected individual
  sp_naive <- y_w_site %>% 
    mutate(detected = rowSums(across(starts_with("visit")), na.rm = TRUE) > 0)
  
  # Join naive occupancy to points for shapefile export later
  naive_occ_sp <- select(sp_naive, Point, year, detected) %>% 
    pivot_wider(names_from = year, names_glue = "n_occ_{year}", values_from = detected)
  survey_pts_sp <- left_join(survey_pts, naive_occ_sp, by = c("NAME" = "Point"))
  
  # Resume naive occupancy summary by year
  sp_naive <- sp_naive %>%
    group_by(year) %>%
    summarise(naive_occ = round(mean(detected), 3), .groups = "drop")
  html_naive <- kbl(sp_naive) %>%
    kable_styling(bootstrap_options = c("striped", "hover", "condensed", "responsive"), full_width = FALSE)
  save_kable(html_naive, file = paste0("Tables/", sp, "_naive_occupancy_thru_", max(sp_naive$year), ".html"))
  message("Naive occupancy")
  message(paste0(capture.output(sp_naive), collapse = "\n"))

  ## Site covariates
  sp_hab <- filter(habitat, year %in% sp_years)
  ### Ensure order matches detection history
  site_covs <- select(y_w_site, Point, year) %>%
    left_join(sp_hab, by = c("Point", "year")) %>%
    mutate(Point = factor(Point))
  site_covs_summary <- summarize_vars(site_covs, year:burns_10_freq) 
  ### Standardize covariates, if applicable
  site_covs <- site_covs %>%
    mutate(across(year:burns_10_freq, ~ as.numeric(scale(.)))) %>% as.data.frame()
  
  ## Observation covariates
  obs_covs <- expand_grid(select(y_w_site, Point, year), visit = 1:max_visits) %>%
    left_join(select(sp_dat, Point, year, doy, Observer, since_rise_h, visit), 
              by = c("Point", "year", "visit")) %>%
    select(doy:since_rise_h) %>%
    mutate(Observer = factor(Observer))
  obs_covs_summary <- summarize_vars(obs_covs, c(doy, since_rise_h))
  obs_covs <- obs_covs %>%
    mutate(across(c(doy, since_rise_h), ~ as.numeric(scale(.)))) %>% as.data.frame()
  covs_summary <- bind_rows(site_covs_summary, obs_covs_summary)
  
  # Create the unmarkedFrame 
  umf_occ <- unmarkedFrameOccu(y = y, #Detection history
                               siteCovs = site_covs, # State covariates
                               obsCovs = obs_covs) # Detection covariates
  
  # Fit maximum model for inference
  sp_occ_mod <- occu(~ doy + since_rise_h + (1|Observer) 
                     ~ year + pine_ba + hard_ba + pine_dbh + snags + canopy + 
                       grass + forb + woody + bare + litter + shrubs + 
                       burns_10_freq # + (1|Point)
                     , data = umf_occ)
  attributes(sp_occ_mod)$var_summary <- covs_summary
  
  # Add occupancy estimates to survey point shapefile
  yr_mean <- filter(covs_summary, variable == "year") %>% pull(mean)
  yr_sd <- filter(covs_summary, variable == "year") %>% pull(sd)
  est_occ_sp <- select(site_covs, Point, year) %>%
    mutate(year = year * yr_sd + yr_mean) %>%
    bind_cols(predict(sp_occ_mod, type = "state")) %>%
    select(Point:Predicted) %>%
    pivot_wider(names_from = year, names_glue = "e_occ_{year}", values_from = Predicted)
  survey_pts_sp <- left_join(survey_pts_sp, est_occ_sp, by = c("NAME" = "Point"))
  shp_nm <- paste0(sp, "_occupancy_thru_2020.shp") # Update year in fn as necessary
  st_write(survey_pts_sp, file.path("Geodata", shp_nm))
  message("Shapefile of naive and estimate occupancy for ", sp, " output to `Geodata` directory.")
  
  # Output summary table 
  html_cov_means <- kbl(covs_summary) %>%
    kable_styling(bootstrap_options = c("striped", "hover", "condensed", "responsive"), full_width = FALSE)
  save_kable(html_cov_means, file = paste0("Tables/", sp, "_covariate_means_", 
                                           paste(range(sp_years), collapse = "-"), ".html"))
  sp_occ_mod
})
names(spp_mods) <- spp
saveRDS(spp_mods, "Output/CSNWR_occupancy-models_2012-2020.rds") # Update year in fn as necessary

# Here is were you'll examine each model and identify the variables to plot for 
# each component of the model (state and detection)

# Bachman's Sparrow
basp_mod <- spp_mods[["BASP"]]
table_occupancy(basp_mod, out_file = "Tables/BASP_occupancy_2012-2020.html") # Update year in fn as necessary

## Detection
basp_det <- c("doy", "since_rise_h")
basp_plots_det <- plot_occupancy(basp_mod, type = "det",
                                 plot_vars = basp_det,
                                 var_labels = label_lookup(basp_det),
                                 var_summary = attr(basp_mod, "var_summary"),
                                 species = "BASP")

## State (occupancy)
basp_state <- c("burns_10_freq", "hard_ba", "bare", "forb") 
basp_plots_state <- plot_occupancy(basp_mod, type = "state",
                                   plot_vars = basp_state,
                                   var_labels = label_lookup(basp_state),
                                   var_summary = attr(basp_mod, "var_summary"),
                                   species = "BASP")

# Brown-headed Nuthatch
bhnu_mod <- spp_mods[["BHNU"]]
table_occupancy(bhnu_mod, out_file = "Tables/BHNU_occupancy_2012-2020.html") # Update year in fn as necessary

## Detection
bhnu_det <- c("since_rise_h", "doy")
bhnu_plots_det <- plot_occupancy(bhnu_mod, type = "det",
                                 plot_vars = bhnu_det,
                                 var_labels = label_lookup(bhnu_det),
                                 var_summary = attr(bhnu_mod, "var_summary"),
                                 species = "BHNU")
 
## State (occupancy)
bhnu_state <- c("hard_ba", "grass", "pine_ba", "pine_dbh", "canopy") 
bhnu_plots_state <- plot_occupancy(bhnu_mod, type = "state",
                                   plot_vars = bhnu_state,
                                   var_labels = label_lookup(bhnu_state),
                                   var_summary = attr(bhnu_mod, "var_summary"),
                                   species = "BHNU")

# Northern Bobwhite
nobo_mod <- spp_mods[["NOBO"]]
table_occupancy(nobo_mod, out_file = "Tables/NOBO_occupancy_2012-2020.html") # Update year in fn as necessary

## Detection
nobo_det <- c("doy", "since_rise_h")
nobo_plots_det <- plot_occupancy(nobo_mod, type = "det",
                                 plot_vars = nobo_det,
                                 var_labels = label_lookup(nobo_det),
                                 var_summary = attr(nobo_mod, "var_summary"),
                                 species = "NOBO")

## State (occupancy)
nobo_state <- c("pine_ba", "burns_10_freq") 
nobo_plots_state <- plot_occupancy(nobo_mod, type = "state",
                                   plot_vars = nobo_state,
                                   var_labels = label_lookup(nobo_state),
                                   var_summary = attr(nobo_mod, "var_summary"),
                                   species = "NOBO")
