# Install and load some package functionality
if (!requireNamespace("pacman", quietly = TRUE)) install.packages("pacman", quiet = TRUE)
if (!requireNamespace("remotes", quietly = TRUE)) install.packages("remotes", quiet = TRUE)
# if (!requireNamespace("nrsmisc", quietly = TRUE))
#   remotes::install_github("adamdsmith/nrsmisc", quiet = TRUE)
pacman::p_load(dplyr, lubridate, chron, readxl, readr, stringr, tidyr, assertr, #nrsmisc, 
               ggplot2, ggrepel, sf, unmarked, knitr, kableExtra)

# Load some created functions
source("Code/Functions/process_pre2020_bird_data.R")
source("Code/Functions/process_post2020_bird_data.R")
source("Code/Functions/process_pre2020_veg_data.R")
source("Code/Functions/process_historic_fire_data.R")
source("Code/Functions/flag_odd_effort.R")
source("Code/Functions/veg_boxplots.R")
source("Code/Functions/veg_point_trends.R")
source("Code/Functions/plot_effort.R")
source("Code/utils.R")

# Load point locations
survey_pts <- st_read("Geodata/CSNWR_PineGrasslandPoints.shp")

# Load burn units
burn_units <- st_read("Geodata/CSNWR_Rx_Unit_Master.shp", quiet = TRUE) %>%
  select(bu_name:COMP)

# Set focal species (codes) to generate final data
focal_spp <- c("BHNU", "BASP", "NOBO")

# Get the point count data for archival surveys (pre-2020)
pre2020_birds <- process_pre2020_bird_data("Data_raw/CSNWR_PineGrasslandBirdData_pre2020.xlsx", flag_odd = FALSE)

# Get the point count data for more recent, digitally-collected surveys (2020 and forward)
post2020_birds <- process_post2020_bird_data("Data_raw/CSNWR_PineGrasslandBirdData_post2020.csv")

# Combine bird data
bird_dat <- bind_rows(pre2020_birds, post2020_birds)

# Pull survey years for fire variable calculation
survey_yrs <- sort(unique(year(bird_dat$DateTime)))

# Plot your effort for most recent year to confirm your expectations
# The following ocde assumes that each point is surveyed 4 times per year for 
# each species. However, that assumption can be changed by adjusting the values 
# for `expected_effort` object created below. Simply enter the number of expected 
# surveys for BACS, BHNU, and NOBO, in that order, followed by the letter "L"
# (the "L" may be confusing, but it simply indicates that a number is an integer).
# For example, if BHNU were only surveyed three times, you would change
# the following line to: 
# expected_effort <- c(BASP = 4L, BHNU = 3L, NOBO = 4L)
expected_effort <- c(BASP = 4L, BHNU = 4L, NOBO = 4L)
# Note the following function has a `year` argument as well to check historical surveys
# flag_odd_effort(bird_dat, expected_effort, year = 2018)
flag_odd_effort(bird_dat, expected_effort)

# Get the vegetation data for archival surveys (pre-2020)
pre2020_veg <- process_pre2020_veg_data("Data_raw/CSNWR_PineGrasslandVegData_pre2020.xlsx")

# Get the vegetation data for more recent, digitally-collected surveys (2024 and forward)
# post2020_veg <- process_post2020_bird_data("Data_raw/CSNWR_PineGrasslandVegData_post2020.csv")

# Combine vegetation data
# veg_dat <- bind_rows(pre2020_veg, post2020_veg)
veg_dat <- pre2020_veg

# Create boxplots of vegetation data
# Note the following function has a `year` argument as well to check historical surveys
# veg_boxplots(veg_dat, year = 2012)
veg_boxplots(veg_dat)

# Explore change in vegetation metrics by survey point
# Potentially unusual changes highlighted for closer inspection
veg_point_trends(veg_dat)

# Calculate or derive variables for joins or additional covariates of interest
veg_dat <- veg_dat %>%
  mutate(year = year(Date), .after = Point) %>%
  select(-Date) %>%
  arrange(Point, year)

# Fill intervening years, recycling data until it changes...
veg_dat <- veg_dat %>%
  complete(Point, year = survey_yrs) %>%
  fill(pine_ba:shrubs, .direction = "down") %>%
  arrange(year, Point)

# Calculate burn frequency information
fire_dat <- process_historic_fire_data("Data_raw/CSNWR_FireFreq_2000_2021.xlsx")
unknown_bu <- flag_unknown_bu(fire_dat, burn_units) %>% pull(bu_name)

burn_history <- full_join(burn_units, fire_dat, by = "bu_name") %>%
  filter(!is.na(fy_burn))

# Create FY burn shapefile
process_fy_burns(burn_history, burn_units, survey_yrs)

# Calculate survey year burn proportion/frequency in 100-m circle around each survey point
# in 10-year intervals inclusive of survey years
survyr_burn_freqs <- calc_survyr_burns(burn_units, burn_history, survey_pts, 
                                       survey_yrs, buffer = 100, year_win = 10) %>%
  arrange(yr_assessed, Point)

# Join habitat and fire
habitat <- left_join(veg_dat, survyr_burn_freqs, by = c("Point", "year" = "yr_assessed"))

# Covariate panels of habitat and fire covariates
select(habitat, pine_ba:burns_10_freq) %>%
  pairs.panels(main = "Habitat and fire covariates")

# Finalize covariates for bird survey data and collapse abundance to occupancy
bird_dat <- bird_dat %>%
  mutate(year = year(DateTime), 
         doy = yday(DateTime), .after = DateTime,
         det = ifelse(Total > 0, 1, 0)) %>%
  add_sun()

# Identify species and year combinations for analysis
(occ_plan <- select(bird_dat, Species, year) %>%
  distinct() %>% arrange(Species, year))
